package jadean.dean.java;


import jadean.dean.Project;
import jadean.dean.Resource;
import jadean.dean.java.constants.JavaProjectConstants;
import jadean.dean.java.resourceparser.jarparser.DefaultJarEntryFilter;
import jadean.dean.java.resourceparser.jarparser.DefaultJarEntryFormatter;
import jadean.dean.java.resourceparser.jarparser.JarResourceUtilities;
import jadean.dean.java.resourceparser.javaparser.visitors.HasMainMethodVisitor;
import jadean.dean.java.resources.JavaResourceContext;
import jadean.dean.java.resources.JavaResourceContextFactory;
import jadean.dean.java.resources.JavaResourceFactory;
import jadean.dean.java.resources.JavaResourceProjectClass;
import jadean.dean.java.resources.JavaResourceProjectResource;
import jadean.dean.java.utilities.FileUtilities;
import jadean.dean.java.utilities.JavaUtilities;
import jadean.dean.java.utilities.ProjectUtilities;
import japa.parser.JavaParser;
import japa.parser.ParseException;
import japa.parser.ast.CompilationUnit;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.jgrapht.Graph;
import org.jgrapht.graph.DefaultDirectedGraph;
import org.jgrapht.graph.DefaultEdge;

public class JavaProject extends Project {
	private List<String> _sourceFolders;
	private String _outputFolder;
	private List<JavaPackage> _referencedPackages;
	private HashMap<JavaPackage, List<String>> _classesInReferencedPackages;
	private HashSet<JavaResource> _visited;
	private boolean _reflectionUsed;

	public JavaProject(String projectName, String[] sourceFolders, String outputFolder, String[] referencedPackages) {
		super(projectName);
		this._sourceFolders = Arrays.asList(sourceFolders);
		this._outputFolder = outputFolder;
		this._referencedPackages = new ArrayList<JavaPackage>();
		for (int i = 0; i < referencedPackages.length; i++) {
			JavaPackage jPackage = new JavaPackage(referencedPackages[i]);
			_referencedPackages.add(jPackage);
			if (jPackage.hasClassPathPackages()) {
				_referencedPackages.addAll(jPackage.getClassPathPackages());
			}
		}
		this.initClassesInReferencedPackages();
		this._reflectionUsed = false;
	}

	public List<String> getSourceFolders() {
		return this._sourceFolders;
	}

	public String getOutputFolder() {
		return this._outputFolder;
	}

	public List<JavaPackage> getReferencedPackages() {
		return this._referencedPackages;
	}

	public HashMap<JavaPackage, List<String>> getClassesInReferencedLibraries() {
		return this._classesInReferencedPackages;
	}

	private void initClassesInReferencedPackages() {
		this._classesInReferencedPackages = null;
		try {
			this._classesInReferencedPackages = JarResourceUtilities.listFiles(this._referencedPackages, new DefaultJarEntryFilter(), new DefaultJarEntryFormatter());
		}
		catch (IOException e) {
			this._classesInReferencedPackages = new HashMap<JavaPackage, List<String>>();
		}
	}
	
	public boolean isContainedInReferencedPackages(String className) {
		for (JavaPackage referencedPackage: _classesInReferencedPackages.keySet()) {
			for (String referencedClassName: _classesInReferencedPackages.get(referencedPackage)) {
				if (referencedClassName.equals(className)) {
					return true;
				}
			}
		}
		return false;
	}
	
	public boolean isContainedInProjectFolders(String className) {
		String fullClassFileName = className.replace('.', File.separatorChar);
		int atomicClassNamePosition = fullClassFileName.lastIndexOf(File.separatorChar);
		String classFilePath = (atomicClassNamePosition == -1) ? "" : fullClassFileName.substring(0, atomicClassNamePosition + 1);
		String classFileName = (atomicClassNamePosition == -1) ? fullClassFileName : fullClassFileName.substring(atomicClassNamePosition + 1);
		for (String sourceFolder: this._sourceFolders) {
			File javaSourceFile = new File(sourceFolder + File.separatorChar + fullClassFileName + ".java");
			if (javaSourceFile.exists())
				return true;
		}
		File javaClassFile = new File(_outputFolder + File.separatorChar + fullClassFileName + ".class");
		if (javaClassFile.exists())
			return true;
		String innerClassFileName = _outputFolder + File.separatorChar + classFilePath + "$" + classFileName + ".class";
		String innerClassDirectory = innerClassFileName.substring(0, innerClassFileName.lastIndexOf(File.separatorChar) + 1);
		String innerClassName = innerClassFileName.substring(innerClassFileName.lastIndexOf(File.separatorChar) + 1);
		if (FileUtilities.innerClassInDir(innerClassDirectory, innerClassName)) {
			return true;
		}
		return false;
		
	}

	public int getResourceType(String className, JavaResourceContext context) {
		if (className.endsWith(".*")) {
			return returnPackageType(className, context);
		}
		else {
			int type = returnProjectClassType(className, context);
			if (type != JavaProjectConstants.RESOURCE_TYPE_UNKNOWN) {
				return type;
			}
			return returnReferencedClassType(className, context);
		}
	}

	private int returnReferencedClassType(String className, JavaResourceContext context) {
		/*
		 * ak je to atomicka trieda, spoji sa so vsetkymi importovanymi packagami a prejde cez
		 * vsetky referencovane package a triedy v nich
		 * ak je to fully qualified trieda, prejdu sa vsetky referencovane package a triedy v nich
		 */
		
		if (JavaUtilities.isAtomicClassName(className)) {
			for (String importedPackage: context.getClassImportedPackages()) {
				// prejde vsetky referencovane package a triedy v nich
				if (isContainedInReferencedPackages(JavaUtilities.buildFullyQualifiedClassName(importedPackage, className))) {
					return JavaProjectConstants.RESOURCE_TYPE_REFERENCED_PACKAGE_CLASS;
				}
			}
			for (String importedClass: context.getClassImportedClasses()) {
				if (JavaUtilities.extractClassName(importedClass).equals(className))
					if (isContainedInReferencedPackages(importedClass))
						return JavaProjectConstants.RESOURCE_TYPE_REFERENCED_PACKAGE_CLASS;
			}
		}
		else {
			// prejde vsetky referencovane package a triedy v nich
			if (isContainedInReferencedPackages(className)) {
				return JavaProjectConstants.RESOURCE_TYPE_REFERENCED_PACKAGE_CLASS;
			}
		}
		
		// classpath class
		return JavaProjectConstants.RESOURCE_TYPE_CLASSPATH_PACKAGE_CLASS;
	}

	private int returnProjectClassType(String className, JavaResourceContext context) {
		/*
		 * ak je to atomicka trieda, spoji sa so vsetkymi importovanymi packagami a prejde cez
		 * vsetky projektove triedy
		 * ak je to FQ trieda, prejdu sa vsetky projektove triedy
		 */
		
		if (JavaUtilities.isAtomicClassName(className)) {
			if (isContainedInProjectFolders(className))
				return JavaProjectConstants.RESOURCE_TYPE_PROJECT_CLASS;
			for (String importedPackage: context.getClassImportedPackages()) {
				if (isContainedInProjectFolders(JavaUtilities.buildFullyQualifiedClassName(importedPackage, className))) 
					return JavaProjectConstants.RESOURCE_TYPE_PROJECT_CLASS;
			}
			for (String importedClass: context.getClassImportedClasses()) {
				if (JavaUtilities.extractClassName(importedClass).equals(className))
					if (isContainedInProjectFolders(importedClass))
						return JavaProjectConstants.RESOURCE_TYPE_PROJECT_CLASS;
			}
		}
		else {
			if (isContainedInProjectFolders(className))
				return JavaProjectConstants.RESOURCE_TYPE_PROJECT_CLASS;	
		}
		return JavaProjectConstants.RESOURCE_TYPE_UNKNOWN;
	}

	private int returnPackageType(String className, JavaResourceContext context) {
		String res = className.replace(".*", "");
		for (String sourceFolder: this._sourceFolders) {
			File f1 = new File(sourceFolder + File.separatorChar + res.replace('.', File.separatorChar));
			if (f1.exists()) return JavaProjectConstants.RESOURCE_TYPE_PROJECT_PACKAGE;
		}
		File f2 = new File(_outputFolder + File.separatorChar + res.replace('.', File.separatorChar));
		if (f2.exists()) return JavaProjectConstants.RESOURCE_TYPE_PROJECT_PACKAGE;
		for (JavaPackage key: _classesInReferencedPackages.keySet()) {
			for (String value: _classesInReferencedPackages.get(key)) {
				if (value.startsWith(res)) return JavaProjectConstants.RESOURCE_TYPE_REFERENCED_PACKAGE;
			}
		}
		return JavaProjectConstants.RESOURCE_TYPE_CLASSPATH_PACKAGE;
	}

	public JavaPackage getPackageForResource(JavaResource resource) {
		if (resource instanceof JavaResourceProjectResource) {
			return ((JavaResourceProjectResource) resource).getPackageForResource();
		}
		return null;
	}

	private void buildDependencyGraph(DefaultDirectedGraph<Resource, DefaultEdge> g, JavaResource resource) {
		if (this._visited.contains(resource)) return;
		Collection<Resource> resources = resource.getReferrencedResources();
		g.addVertex(resource);
		this._visited.add(resource);
		for (Resource _r: resources) {
			JavaResource r = (JavaResource) _r;
			if (resource.equals(r)) continue;
			g.addVertex(r);
			g.addEdge(resource, r);
			this.buildDependencyGraph(g, r);
		}		
	}

	@Override
	public Graph<Resource, DefaultEdge> getDependencyGraph(String entryPoint) {
		JavaResource javaResourceEntryPoint = new JavaResourceProjectClass(entryPoint, JavaResourceContextFactory.getResourceContext(this, entryPoint), this);
		DefaultDirectedGraph<Resource, DefaultEdge> ret = new DefaultDirectedGraph<Resource, DefaultEdge>(DefaultEdge.class);
		this._visited = new HashSet<JavaResource>();
		this.buildDependencyGraph(ret, javaResourceEntryPoint);
		return ret;
	}
	
	@Override
	public Collection<Resource> getUsedResources(String entryPoint) {
		JavaResource javaResourceEntryPoint = new JavaResourceProjectClass(entryPoint, JavaResourceContextFactory.getResourceContext(this, entryPoint), this);
		DefaultDirectedGraph<Resource, DefaultEdge> ret = new DefaultDirectedGraph<Resource, DefaultEdge>(DefaultEdge.class);
		this._visited = new HashSet<JavaResource>();
		this.buildDependencyGraph(ret, javaResourceEntryPoint);
		return ProjectUtilities.removeResourcesFromPackage(ret.vertexSet());
	}
	
	@Override
	public Collection<Resource> getAllResources() {
		Set<Resource> ret = new HashSet<Resource>();
		JavaResourceFactory rf = new JavaResourceFactory(this);
		for (JavaPackage referencedPackage: this._referencedPackages) {
			try {
				List<String> resources = referencedPackage.getAllResources();
				for (String s: resources) {
					JavaResourceContext cnc = new JavaResourceContext();
					ret.add(rf.classNameToResource(s, cnc));
				}				
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		for (File f: FileUtilities.allFilesInDirs(this.getSourceFolders())) {
			String className = FileUtilities.getClassName(f, this.getSourceFolders());
			JavaResourceContext context = JavaResourceContextFactory.getResourceContext(this, className);
			ret.add(rf.classNameToResource(className, context));
		}
		return ret;
	}
	
	@Override
	public Collection<Resource> getUnusedResources(String entryPoint) {
		Collection<Resource> usedResources = this.getUsedResources(entryPoint);
		Collection<Resource> allResources = this.getAllResources();
		allResources.removeAll(usedResources);
		Set<Resource> unusedResources = new HashSet<Resource>(allResources);
		for (Resource usedResource: usedResources) {
			if (usedResource.getName().endsWith(".*")) {
				for (Resource resource: allResources) {
					if (resource.getName().startsWith(usedResource.getName().replace(".*", ""))) {
						unusedResources.remove(resource);
					}
				}
			}
		}
		return unusedResources;
	}
	
	public Collection<JavaResourceProjectClass> getClassesWithMainMethod() {
		Set<JavaResourceProjectClass> mainClasses = new HashSet<JavaResourceProjectClass>();
		
		List<File> files = FileUtilities.allFilesInDirs(this.getSourceFolders());
		for (File f: files) {
			HasMainMethodVisitor<Void> hmmv = new HasMainMethodVisitor<Void>();
			CompilationUnit cu = null;
			try {
				cu = JavaParser.parse(f);
			} catch (ParseException e) {
			} catch (IOException e) {
			}
			if (cu != null) {
				hmmv.visit(cu, null);
				if (hmmv.hasMainMethod()) {
					mainClasses.add(new JavaResourceProjectClass(FileUtilities.getClassName(f, this.getSourceFolders()), null, null));
				}
			}
		}
		
		return mainClasses;
	}

	@Override
	public void removeResources(Collection<Resource> resources) {
		Set<JavaResource> javaResources = new HashSet<JavaResource>();
		
		for (Resource r: resources) {
			javaResources.add((JavaResource) r);
		}		
		for (JavaPackage jp: this.getReferencedPackages()) {
			jp.removeResources(javaResources);
		}
		
		for (JavaResource jr: javaResources) {
			jr.delete();
		}
		
	}
	
	public void setReflectionUsed(boolean b) {
		this._reflectionUsed = b;
	}
	
	public boolean isReflectionUsed() {
		return this._reflectionUsed;
	}

	public String extractResourceNameFromFileName(String fileName) {
		for (String s : this.getSourceFolders()) {
			if (fileName.startsWith(s)) {
				return fileName.replace(s + File.separatorChar, "").replace(".java", "").replace(File.separatorChar, '.');
			}
		}
		return "";
	}
}
